// Adobe.FaceTracker.js
/*global define, console */
/*jslint sub: true */ 

define([ "lib/Zoot", "src/utils", "src/math/Vec2", "src/math/mathUtils", "lib/dev", "lib/tasks", "lodash", "src/animate/faceUtils" ],
  function (Zoot, utils, vec2, mathUtils, dev, tasks, lodash, faceUtils) {
		"use strict";
			
	// first three are default mouth shapes (match default mouth shape classifiers in beaker::zoot:FaceClassifiers)
	// TODO: eventually allow users to add to this list with appropriate named mouth shapes for customized facial expression recognition
	var mouthShapeLabels = [
			"Adobe.Face.NeutralMouth",
			"Adobe.Face.SurprisedMouth",		// warning: the indexes of these must match FaceClassifiers::GetMouthShape2DSimple()
			"Adobe.Face.SmileMouth"
		],

		// sequences of fall back mouth shape indices
		// whichever mouth shape index the facetracker returns, 
		// chooseMouthReplacements will use the first valid mouth shape in the corresponding sequence
		mouthShapeCascades = {
			0: [0], 
			1: [1, 0],
			2: [2, 0]

//			3: [3, 1, 0], commented out along with "Grimace" - "Tongue" above
//			4: [4, 0],
		};



	function defineMouthLayerParams () {
		var aParams = [];
		
		faceUtils.mouthShapeLayerTagDefinitions.forEach( function (tagDefn) {
			aParams.push({id:faceUtils.makeLayerIdFromLabel(tagDefn.id), type:"layer", uiName:tagDefn.uiName,
						dephault:{match:"//"+tagDefn.id,  startMatchingAtParam:"mouthsParent"},
						maxCount:1}); // only one of each mouth shape per "Mouth" group
		});
		
		var mouthParentTagDefn = faceUtils.mouthParentLayerTagDefinition[0];
		// add this to the front, after sorting the rest
		aParams.splice(0, 0, {id:"mouthsParent", // TODO: use startMatchingAtParam: "viewLayers" but first need support for more than one sMAP
							type:"layer",
							uiName:mouthParentTagDefn.uiName,
							dephault:{match:"//"+mouthParentTagDefn.id}});		
		
		return aParams;
	}


	// TODO: factor with LipSync.js
	// returns array of maps from mouth label to matched layer, one array per mouth group
	// called during onCreate, so it uses getStaticParam
	//	outer array is one per mouth group (i.e. number of matches of mouthsParent)
	function getMouthsCandidates (args) {
		var aMouthsLayers = [],
			bHideSiblings = true,
			aMouths = args.getStaticParam("mouthsParent");
		
		aMouths.forEach(function (mouthLayer, mouthGroupIndex) {
			var mouths = { parent: mouthLayer };	// map from label -> layer
			mouthShapeLabels.forEach(function (label) {
				var aaLayers = args.getStaticParam(faceUtils.makeLayerIdFromLabel(label)),
					layer = aaLayers[mouthGroupIndex][0]; // we only support one of each shape per mouth
				
				if (layer) {
					mouths[label] = layer;
					layer.setTriggerable(bHideSiblings);
				}
			});
			aMouthsLayers.push(mouths);
		});
		
		return aMouthsLayers;
	}

	
	// returns actual mouth node for requested mouth index, which may not exist (uses fallbacks in mouthShapeCascades)
	function getClosestValidMouthContainerRoot(args, groupIndex, inMouthIndex) {

		var mouthIndicesToTry = [], i, mouthIndex, mouthShapeLabel,
			aLayers, layer;

		if (mouthShapeCascades.hasOwnProperty(inMouthIndex)) {
			mouthIndicesToTry = mouthShapeCascades[inMouthIndex];
		}
		else {
			mouthIndicesToTry = [inMouthIndex, 0];
		}

		for (i = 0; i < mouthIndicesToTry.length; i += 1) {
			mouthIndex = mouthIndicesToTry[i];
			if (mouthIndex >= 0 && mouthIndex < mouthShapeLabels.length) {
				mouthShapeLabel = mouthShapeLabels[mouthIndex];
				if (!mouthShapeLabel) {
					// this shouldn't happen, so adding error message if it does to help diagnose
					console.log("invalid mouth index: " + mouthIndex);
				} else {
					aLayers = args.getParam(faceUtils.makeLayerIdFromLabel(mouthShapeLabel))[groupIndex];
					layer = aLayers[0];

					if (layer) {
						return layer;
					}
				}
			}
		}

		return null;
	}

	
	function chooseMouthReplacements (args, aMouthsGroups, head14) {
		var priority = 0.25,
			mouthIndexToShow = Math.max(head14["Head/MouthShape"], 0);
		
		if (mouthIndexToShow >= 0) {
			aMouthsGroups.forEach(function (mouthsGroup, groupIndex) {
				// get valid (found) mouth that is closest to specified mouth to show
				var validMouthLayer = getClosestValidMouthContainerRoot(args, groupIndex, mouthIndexToShow);

				if (validMouthLayer) {
					validMouthLayer.trigger(priority); // compare priority to LipSync & KeyReplacer
				} else {
					// alternatively, we could output error msg saying that we can't find a Neutral mouth
					//console.logToUser("choseMouthReplacements(): no Neutral mouth found");
				}
			});
		}
	}

	function chooseBlinkReplacements(aaLayers, eyeOpenness, threshold, eyeFactor) {
		aaLayers.forEach(function (aLays) {
			aLays.forEach(function (blinkLayer) {
				if ((eyeOpenness <= threshold) && (eyeFactor !== 0))  { // the eyelid measurment in head14 never seems to go below about .3
					blinkLayer.trigger();
				}
			});
		});
	}

	function setAllLayersAsTriggerableHideSibs(aaLays) {
		var bHideSiblings = true;

		aaLays.forEach(function (aLays) {
			aLays.forEach(function (lay) {
				lay.setTriggerable(bHideSiblings);
			});
		});
	}

	function setTriggerableLayers(self, args) {
		
		setAllLayersAsTriggerableHideSibs(self.leftBlinkLayers);

		setAllLayersAsTriggerableHideSibs(self.rightBlinkLayers);

		// gather replacement mouth layers, also sets as triggerable
		self.aMouthsGroups = getMouthsCandidates(args); // array of maps from label -> layer, and .parent too
	}

	/*function printPuppetContainerHandleHierarchies(inPuppet) {

		inPuppet.breadthFirstEach(function (p) {
			var cTree, hTree;
			console.log("p = " + p.getName());
			cTree = p.getContainerTree();

			if (cTree) {
				cTree.breadthFirstEach(function (c) {
					console.log("\tc = " + c.getName());
				});
			}

			hTree = p.getHandleTreeRoot();
			if (hTree) {
				console.log("\thTree:");				
				hTree.breadthFirstEach(function (h) {
					console.log("\t\th = " + h.getName());
				});
			}
		});
	}*/

	function chooseFaceTrackerReplacements(self, args, head14) {
		var eyeFactor = args.getParam("eyeFactor"),
			eyeClosedThresh = 0.45;
		
		chooseMouthReplacements(args, self.aMouthsGroups, head14);
		
		var leftEyelid = head14["Head/LeftEyelid"],
			rightEyelid = head14["Head/RightEyelid"];
		
		chooseBlinkReplacements(self.leftBlinkLayers, leftEyelid, eyeClosedThresh, eyeFactor);
		chooseBlinkReplacements(self.rightBlinkLayers, rightEyelid, eyeClosedThresh, eyeFactor);
	}

	function animateWithFaceTracker(self, args) {
		var head14 = faceUtils.getFilteredHead14(args), 
			leftEyelid, rightEyelid;

		// this could be done within faceUtils as well ...
		if (args.getParam("blinkOnly")) {
			leftEyelid = head14["Head/LeftEyelid"];
			rightEyelid = head14["Head/RightEyelid"];
			head14["Head/LeftEyelid"] = head14["Head/RightEyelid"] = (leftEyelid + rightEyelid) / 2;
		}

		// if we end up putting mouths under views, this will need to go into the loop just below
		chooseFaceTrackerReplacements(self, args, head14);
		
		self.aViews.forEach(function (view, viewIndex) {		
			var transforms = faceUtils.computePuppetTransforms(self, args, head14, viewIndex);
			
			// Handled by Eye Gaze behavior. 
			delete transforms["Adobe.Face.LeftPupil"];
			delete transforms["Adobe.Face.RightPupil"];
			
			faceUtils.applyPuppetTransforms(self, args, viewIndex, transforms);
		});
	}
	
	return {
		about:			"$$$/private/animal/Behavior/FaceTracker/About=Face Tracker, (c) 2015.",
		description:	"$$$/animal/Behavior/FaceTracker/Desc=Controls head, eyes, eyebrows, nose, and mouth via your webcam",
		uiName:			"$$$/animal/Behavior/FaceTracker/UIName=Face",
		defaultArmedForRecordOn: true,
	
		defineParams: function () { // free function, called once ever; returns parameter definition (hierarchical) array
		  return [
			faceUtils.cameraInputParameterDefinitionV2, 
			{ id: "poseFilteringLevel", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/poseFilteringLevel=Pose-to-Pose Movement", uiUnits: "%", min: 0, max: 100, precision: 0, dephault: 0, hideRecordButton: true,
				uiToolTip: "$$$/animal/behavior/face/param/poseFilteringLevel/tooltip=Exaggerate or minimize pausing the face or head at key poses; adjust Smoothing to ease the transition between poses" },
			{ id: "minPoseDuration", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/minPoseDuration=Minimum Pose Duration", uiUnits: "$$$/animal/InspectScene/units/sec=sec", precision:1, min:0, dephault:1, hideRecordButton: true,
				uiToolTip: "$$$/animal/behavior/face/param/minPoseDuration/tooltip=Limits how quickly key poses can happen when Pose-to-Pose Movement is enabled" },
			{ id: "smoothingRate", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/smoothingRate=Smoothing", uiUnits: "%", min: 0, max: 100, precision: 0, dephault: 15, hideRecordButton: true,
				uiToolTip: "$$$/animal/behavior/face/param/smoothingRate/tooltip=Exaggerate or minimize the smoothness when transitioning between different face or head poses" },
			{ id: "headPosFactor", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/headPosFactor=Head Position Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 100,
				uiToolTip: "$$$/animal/behavior/face/param/headPosFactor/tooltip=Exaggerate or minimize how much the head moves when you move your head" },
			{ id: "headScaleFactor", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/headScaleFactor=Head Scale Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 25,
				uiToolTip: "$$$/animal/behavior/face/param/headScaleFactor/tooltip=Exaggerate or minimize how much the head scales when you move your head closer or farther from the camera" },
			{ id: "headRotFactor", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/headRotFactor=Head Tilt Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 75,
				uiToolTip: "$$$/animal/behavior/face/param/headRotFactor/tooltip=Exaggerate or minimize how much the head rotates when you tilt the top of your head to the right and left" },
			{ id: "eyebrowFactor", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/eyebrowFactor=Eyebrow Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 75,
				uiToolTip: "$$$/animal/behavior/face/param/eyebrowFactor/tooltip=Exaggerate or minimize how much the eyebrows move when you raise and lower your eyebrows" },
			{ id: "eyeFactor", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/eyeFactor=Eyelid Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 100,
				uiToolTip: "$$$/animal/behavior/face/param/eyeFactor/tooltip=Exaggerate or minimize how far the eyelids move (if present) or the eyes scale (if not) when you blink" },
			{ id: "mouthFactor", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/mouthFactor=Mouth Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 0,
				uiToolTip: "$$$/animal/behavior/face/param/mouthFactor/tooltip=Exaggerate or minimize how much the mouth scales and moves to correspond with your mouth movements" },
			{ id: "parallaxFactor", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/parallaxFactor=Parallax Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 100,
				uiToolTip: "$$$/animal/behavior/face/param/parallaxFactor/tooltip=Exaggerate or minimize the eyes and nose movement when you rotate your head left and right" },
			// TODO: set dephault values below to make eyebrow motion look natural
			{ id:"upEyebrowsTilt", type:"slider", uiName:"$$$/animal/Behavior/FaceTracker/Parameter/upEyebrowsTilt=Raised Eyebrow Tilt", uiUnits: "%", precision:0, dephault:20, 
				uiToolTip: "$$$/animal/Behavior/FaceTracker/Parameter/upEyebrowsTilt/tooltip=Amount of eyebrow tilt at their highest position" },			
			{ id:"downEyebrowsTilt", type:"slider", uiName:"$$$/animal/Behavior/FaceTracker/Parameter/downEyebrowsTilt=Lowered Eyebrow Tilt", uiUnits: "%", precision:0, dephault:-20, 
				uiToolTip: "$$$/animal/Behavior/FaceTracker/Parameter/downEyebrowsTilt/tooltip=Amount of eyebrow tilt at their lowest position" },						
			{ id: "moveEyebrowsTogether", type: "checkbox", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/moveEyebrowsTogether=Move Eyebrows Together", dephault: true },
			{ id: "viewLayers", type: "layer", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/Views=Views", dephault: { match:["//Adobe.Face.LeftProfile|Adobe.Face.LeftQuarter|Adobe.Face.Front|Adobe.Face.RightQuarter|Adobe.Face.RightProfile|Adobe.Face.Upward|Adobe.Face.Downward", "."] } },
			{ id: "handlesGroup", type: "group", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/handlesGroup=Handles", groupChildren: faceUtils.defineHandleParams(false, true) },
			{ id: "replacementsGroup", type: "group", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/replacementsGroup=Replacements", groupChildren: [
				{ id: "leftBlinkLayers", type: "layer", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/leftBlinkLayers=Left Blink", dephault: { match: "//Adobe.Face.LeftBlink", startMatchingAtParam: "viewLayers" }, maxCount: 1 },
				{ id: "rightBlinkLayers", type: "layer", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/rightBlinkLayers=Right Blink", dephault: { match: "//Adobe.Face.RightBlink", startMatchingAtParam: "viewLayers" }, maxCount: 1 },
				{ id: "blinkOnly", type: "checkbox", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/blinkOnly=Blink Eyes Together", dephault: true },
				{ id: "mouthGroup", type: "group", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/mouthGroup=Mouths", groupChildren: defineMouthLayerParams() }
			]},

			{ id: "poseToPoseGroup", type: "group", uiName: "PoseToPose", hidden: true, groupChildren: [
				// NOTE: setting whole group to be hidden should hide the items below, but that appears not to be working at the moment; if/when that gets fixed, we can remove the hidden settings below
				{ id: "enablePoseToPoseAdvancedParams", type: "checkbox", uiName: "Enable Pose-to-Pose Advanced", dephault:false, hidden: true,
					uiToolTip: "Enable pose-to-pose filter with advanced parameters" },
				{ id: "filterNumSamples", type: "slider", uiName: "Num Samples", precision:1, min:1, dephault:30, hidden: true,
					uiToolTip: "Number of samples (at scene frame rate) in pose-to-pose filter" },
				{ id: "minInterval", type: "slider", uiName: "Min Interval For Switch", uiUnits: "s", precision:3, min:0, dephault:0.1, hidden: true,
					uiToolTip: "Minimum interval in seconds between pose switches" },
				{ id: "maxInterval", type: "slider", uiName: "Max Interval For Switch", uiUnits: "s", precision:3, min:0, dephault:3, hidden: true,
					uiToolTip: "Maximum interval in seconds between pose switches" },
				{ id: "headRotMeanThresh", type: "slider", uiName: "Head Rotation Mean Thresh", precision:3, min:0, dephault:0.25, hidden: true,
					uiToolTip: "Threshold to trigger pose transitions for mean head rotation values" },
				{ id: "headRotVarThresh", type: "slider", uiName: "Head Rotation Var Thresh", precision:3, min:0, dephault:0.01, hidden: true,
					uiToolTip: "Threshold to trigger pose transitions for variance of head rotation values" },
				{ id: "headPosMeanThresh", type: "slider", uiName: "Head Position Mean Thresh", precision:3, min:0, dephault:0.25, hidden: true,
					uiToolTip: "Threshold to trigger pose transitions for mean of head position values" },
				{ id: "headPosVarThresh", type: "slider", uiName: "Head Position Var Thresh", precision:3, min:0, dephault:0.01, hidden: true,
					uiToolTip: "Threshold to trigger pose transitions for variance of head position values" },
				{ id: "eyebrowPosMeanThresh", type: "slider", uiName: "Eyebrow Position Mean Thresh", precision:3, min:0, dephault:0.25, hidden: true,
					uiToolTip: "Threshold to trigger pose transitions for mean of eyebrow position values" },
				{ id: "eyebrowPosVarThresh", type: "slider", uiName: "Eyebrow Position Var Thresh", precision:3, min:0, dephault:0.01, hidden: true,
					uiToolTip: "Threshold to trigger pose transitions for variance of eyebrow position values" }
			]}
		  ];
		},
		
		defineTags: function () {
			var aAllTags = [];
			
			var aAllTagRefs = [faceUtils.headFeatureTagDefinitions, faceUtils.eyeFeatureTagDefinitions, faceUtils.faceFeatureTagDefinitions, faceUtils.eyelidLayerTagDefinitions, faceUtils.mouthShapeLayerTagDefinitions, faceUtils.mouthParentLayerTagDefinition, faceUtils.viewLayerTagDefinitions, faceUtils.miscLayerTagDefinitions];

			aAllTagRefs.forEach(function (ar) {
				aAllTags = aAllTags.concat(ar);
			}); 
			return {aTags:aAllTags};
		},
		
		onCreateBackStageBehavior: function (/*self*/) {
			return { order: 0.51, importance : 0.0 }; // must come after LipSync
		},
		
		onCreateStageBehavior: function (self, args) {
			faceUtils.onCreateStageBehavior(self, args, true, false, true);
			
			// find blink replacements
			self.leftBlinkLayers = args.getStaticParam("leftBlinkLayers");
			self.rightBlinkLayers = args.getStaticParam("rightBlinkLayers");

			setTriggerableLayers(self, args); // not view-multiplexed right now; only mouth-multiplexed

			// debug -- show puppet/container hierarchy
			// printPuppetContainerHandleHierarchies(stagePuppet);
		},

		// Clear the rehearsal state
		onResetRehearsalData : function (self) {
			faceUtils.onResetRehearsalData(self);
		},

		onFilterLiveInputs: function (self, args) { // method on behavior that is attached to a puppet, only onstage
			faceUtils.onFilterLiveInputs(self, args, /*filterEyeGazeB*/ false);
		},
		
		onAnimate: function (self, args) { // method on behavior that is attached to a puppet, only onstage
			animateWithFaceTracker(self, args);
		}
		
	}; // end of object being returned
});
